<?php
// scan.php

// Configurações
const M3U_CANAIS = 'https://xuis.n-1.fun/canais.m3u';
const M3U_LOGOS = 'https://xuis.n-1.fun/logos.m3u';
const TMP_DIR = __DIR__ . '/tmp/';

if(!is_dir(TMP_DIR)) mkdir(TMP_DIR, 0755, true);

// Funções auxiliares (baixar, extrair meta, buscar meta, escanear) - igual código anterior
function baixarUrl($url) {
    $context = stream_context_create(['http' => ['timeout' => 5]]);
    return @file_get_contents($url, false, $context);
}

function extrairMetadados($linha) {
    preg_match('/tvg-logo="([^"]*)".*group-title="([^"]*)",\s*(.*)/i', $linha, $m);
    return [
        'logo' => $m[1] ?? 'https://i.imgur.com/qz6sV6V.png',
        'grupo' => $m[2] ?? 'Sem Grupo',
        'nome' => $m[3] ?? 'Canal Desconhecido'
    ];
}

function processarM3U($url) {
    $raw = baixarUrl($url);
    if (!$raw) return [];
    $linhas = explode("\n", $raw);
    $metadados = [];
    for($i=0; $i<count($linhas); $i++){
        if(strpos($linhas[$i],'#EXTINF')!==false){
            $meta = extrairMetadados($linhas[$i]);
            $link = trim($linhas[$i+1] ?? '');
            if(filter_var($link, FILTER_VALIDATE_URL)){
                $metadados[$link] = $meta;
            }
        }
    }
    return $metadados;
}

function buscarMetadadosPorNome($nome, $lista) {
    $nome = mb_strtolower(trim($nome));
    foreach($lista as $url=>$meta){
        if(mb_stripos($meta['nome'], $nome) !== false) return $meta;
    }
    return null;
}

function escanearCanais($ip, $canais_m3u, $logos_m3u) {
    $portas = range(14000, 15000);
    $encontrados = [];
    foreach ($portas as $porta) {
        $url = "http://$ip:$porta";
        $headers = @get_headers($url);
        if ($headers && strpos($headers[0], '200') !== false) {
            $meta_canal = $canais_m3u[$url] ?? null;
            $nome_canal = $meta_canal['nome'] ?? "Canal $porta";

            $meta_logo = buscarMetadadosPorNome($nome_canal, $logos_m3u);
            if ($meta_logo === null) {
                $meta_logo = ['logo'=>'https://i.imgur.com/qz6sV6V.png', 'grupo'=>'Sem Grupo'];
            }

            $encontrados[] = [
                'nome' => $nome_canal,
                'grupo' => $meta_logo['grupo'],
                'logo' => $meta_logo['logo'],
                'url' => $url
            ];
        }
    }
    return $encontrados;
}

function gravarVideo10s($url, $id) {
    $file = TMP_DIR . "preview_{$id}_" . time() . ".mp4";
    $cmd = "ffmpeg -y -i \"$url\" -t 10 -c copy \"$file\" 2>&1";
    exec($cmd, $output, $ret);
    return $ret === 0 && file_exists($file) ? $file : false;
}

function gerarM3U($lista) {
    $conteudo = "#EXTM3U\n";
    foreach($lista as $canal){
        $logo = $canal['logo'] ?? '';
        $grupo = $canal['grupo'] ?? 'Sem Grupo';
        $nome = $canal['nome'] ?? 'Sem Nome';
        $url = $canal['url'];
        $conteudo .= "#EXTINF:-1 tvg-logo=\"$logo\" group-title=\"$grupo\",$nome\n$url\n";
    }
    return $conteudo;
}

function salvarM3U($conteudo) {
    $file = TMP_DIR . "playlist_" . time() . ".m3u";
    file_put_contents($file, $conteudo);
    return $file;
}

// Se for POST AJAX (escaneando IP)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ip'])) {
    $ip = trim($_POST['ip']);
    header('Content-Type: application/json');

    if (!filter_var($ip, FILTER_VALIDATE_IP)) {
        echo json_encode(['error' => 'IP inválido']);
        exit;
    }

    $canais_m3u = processarM3U(M3U_CANAIS);
    $logos_m3u = processarM3U(M3U_LOGOS);

    $encontrados = escanearCanais($ip, $canais_m3u, $logos_m3u);

    if (count($encontrados) === 0) {
        echo json_encode(['error' => 'Nenhum canal encontrado']);
        exit;
    }

    // Gravar vídeos preview
    $videos = [];
    foreach($encontrados as $i => $canal){
        $video = gravarVideo10s($canal['url'], $i);
        if($video !== false){
            $videos[] = ['nome'=>$canal['nome'], 'video'=>basename($video)];
        }
    }

    // Gerar arquivo M3U e devolver caminho
    $m3u_conteudo = gerarM3U($encontrados);
    $arquivo_m3u = salvarM3U($m3u_conteudo);

    echo json_encode([
        'success' => true,
        'canais' => $encontrados,
        'videos' => $videos,
        'arquivo_m3u' => basename($arquivo_m3u)
    ]);
    exit;
}

// Interface HTML básica
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
<meta charset="UTF-8" />
<title>ZOOMIER TECH | Scanner IPTV</title>
<style>
body { font-family: Arial,sans-serif; background:#121212; color:#eee; margin:20px; }
input[type=text]{ width:300px; padding:8px; border-radius:5px; border:none; margin-right:10px;}
button{padding:8px 15px; background:#0f0; border:none; border-radius:5px; color:#000; cursor:pointer;}
button:hover{background:#0c0;}
.canal-card{background:#222; border-radius:6px; margin:10px 0; padding:10px; display:flex; align-items:center; gap:10px;}
.canal-logo{width:80px; height:45px; object-fit: contain; border-radius:4px; background:#000;}
.video-preview{max-width:320px; margin-top:5px;}
a.download-link{display:inline-block; margin-top:10px; background:#0f0; padding:6px 10px; border-radius:5px; text-decoration:none; color:#000;}
a.download-link:hover{background:#0c0;}
</style>
</head>
<body>

<h1>ZOOMIER TECH | Scanner IPTV</h1>
<p>Digite o IP para escanear as portas 14000-15000:</p>

<input type="text" id="ip" placeholder="Ex: 167.357.377.71" />
<button onclick="startScan()">Escanear</button>

<div id="status"></div>
<div id="resultado"></div>

<script>
function startScan(){
    const ip = document.getElementById('ip').value.trim();
    if(!ip){
        alert('Informe um IP válido!');
        return;
    }
    document.getElementById('status').innerText = '⏳ Escaneando... Aguarde...';
    document.getElementById('resultado').innerHTML = '';

    fetch('scan.php', {
        method: 'POST',
        headers: {'Content-Type':'application/x-www-form-urlencoded'},
        body: 'ip='+encodeURIComponent(ip)
    }).then(res => res.json())
    .then(data => {
        if(data.error){
            document.getElementById('status').innerText = '❌ '+data.error;
            return;
        }
        document.getElementById('status').innerText = '✅ Scan completo! Canais encontrados: ' + data.canais.length;

        let html = '';
        data.canais.forEach((canal, i) => {
            html += `<div class="canal-card">
                <img class="canal-logo" src="${canal.logo}" alt="${canal.nome}" />
                <div>
                    <b>${canal.nome}</b><br/>
                    <small>${canal.grupo}</small><br/>
                    <a href="${canal.url}" target="_blank">Abrir Stream</a><br/>
                    ${data.videos.find(v => v.nome === canal.nome) ? `<video class="video-preview" controls src="tmp/${data.videos.find(v => v.nome === canal.nome).video}"></video>` : ''}
                </div>
            </div>`;
        });
        html += `<a class="download-link" href="tmp/${data.arquivo_m3u}" download>⬇️ Baixar Arquivo M3U Completo</a>`;

        document.getElementById('resultado').innerHTML = html;
    }).catch(e => {
        document.getElementById('status').innerText = '❌ Erro na requisição.';
    });
}
</script>

</body>
</html>